/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2019-05-09 14:25:04 +0000 (Thu, 09 May 2019) $ $Rev: 1592 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.impl;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;

import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.schema.Namespace;
import de.consist.bmu.rule.util.CertUtils;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * @author srossbroich
 * 
 */
public final class RuleImplSignaturePN extends RuleImpl {

    private static final long serialVersionUID = 1L;

    private static final Log LOGGER = LogFactory
            .getLog(RuleImplSignaturePN.class);

    private static final String TECHDOC = "Prft den 'Common Name' (CN) im Subject-DN des Zertifikats "
            + "(ds:Signature/ds:KeyInfo/ds:X509Data/ds:X509Certificate) in allen Signaturen "
            + "im Dokument auf das Vorkommen von ':PN'";
    /**
     * XPath fuer ds:Signature.
     */
    private static final String XPATH_SIGNATURE = "/descendant::ds:Signature[@Id]";

    /**
     * @param ruleDef
     *            RuleDef
     */
    public RuleImplSignaturePN(RuleDef ruleDef) {
        super(ruleDef);
    }

    /**
     * Prft den CN ("2.5.4.3"), nicht auf PSEUDONYM ("2.5.4.65").
     * 
     * @param x509Cert
     *            X509Certificate
     * @return false wenn ':PN' im CommonName (CN), sonst true
     * @throws BMUException
     *             BMUException
     */
    public static boolean isValid(X509Certificate x509Cert) throws BMUException {
        boolean retVal = true;
        String cn = CertUtils.getSubjectCN(x509Cert);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("CN=" + cn);
        }
        if (cn == null || cn.contains(":PN")) {
            retVal = false;
        }
        return retVal;
    }

    /**
     * {@inheritDoc}
     */
    public List<RuleResult> execute(BMUDokument bmuDok) throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        NodeList nl;
        try {
            nl = XPathFassade.getInstance().evaluateNodeList(
                    bmuDok.getDocument(), XPATH_SIGNATURE);
        } catch (XPathExpressionException e) {
            throw new BMUException("Fehler beim Prfen", e);
        }
        int index = 1;
        for (int i = 0; i < nl.getLength(); i++) {
            Node sigNode = nl.item(i);
            if (sigNode instanceof Element) {
                Element sigElem = (Element) sigNode;
                String sigID = sigElem.getAttribute("Id");
                NodeList certNodeList = sigElem.getElementsByTagNameNS(
                        Namespace.xmldsig.getUri(), "X509Certificate");
                if (certNodeList == null || certNodeList.getLength() == 0) {
                    // Ist ein Pflichtfeld, kann nicht vorkommen
                    LOGGER.info("Kein Element <X509Certificate> gefunden. Dies kann nicht vorkommen!");
                } else if (certNodeList.getLength() > 1) {
                    LOGGER.warn("Die Prfung auf Pseudonym (':PN') kann nicht durchgefhrt werden, da mehrere Elemente 'X509Certificate' in der Signatur mit der Id=" + sigID + " vorhanden sind!");
                } else {
                    for (int j = 0; j < certNodeList.getLength(); j++) {
                        try {
                            X509Certificate x509Cert = CertUtils
                                    .getX509CertificateFromBase64(certNodeList
                                            .item(j).getTextContent());
                            if (!isValid(x509Cert)) {
                                ruleResultList.add(new RuleResultImpl(this
                                        .getRuleDef(), index++, sigID));
                            }
                        } catch (Exception e) {
                            LOGGER.error("Error reading X509Certificate", e);
                            throw new BMUException("Error reading X509Certificate",
                                    e);
						}
                    }
                }
            }
        }
        return ruleResultList;
    }

    /**
     * @return Die technische Dokumentation der Implementierung
     */
    public static String getTechDoc() {
        return TECHDOC;
    }

}
