/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-07-22 09:03:03 +0000 (Mon, 22 Jul 2024) $ $Rev: 1829 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.util;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.Key;
import java.security.KeyStore;
import java.security.NoSuchProviderException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Collections;
import javax.security.auth.x500.X500Principal;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
// BC 1.50
//import org.bouncycastle.asn1.ASN1String;
//import org.bouncycastle.asn1.x500.RDN;
//import org.bouncycastle.asn1.x500.X500Name;
//import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils;

import de.consist.bmu.rule.error.BMUException;

/**
 * @author srossbroich
 * 
 */
public final class CertUtils {

	private static final Log LOGGER = LogFactory.getLog(CertUtils.class);

	private CertUtils() {
	}

	/**
	 * @param certData
	 *            byte[]
	 * @return X509Certificate
	 * @throws CertificateException
	 *             CertificateException
	 * @throws NoSuchProviderException 
	 */
	public static X509Certificate getX509Certificate(byte[] certData) throws CertificateException, NoSuchProviderException {
		if (certData == null || certData.length == 0) {
			throw new IllegalArgumentException("certData must not be null or empty");
		}
		X509Certificate x509Cert = null;
		ByteArrayInputStream bais = new ByteArrayInputStream(certData);
		x509Cert = CertUtils.getX509Certificate(bais);
		// Die close()-Methode in ByteArrayInputStream macht nichts, kann
		// entfallen
		// bais.close();
		return x509Cert;
	}

	/**
	 * @param inStream
	 *            InputStream
	 * @return X509Certificate
	 * @throws CertificateException
	 *             CertificateException
	 * @throws NoSuchProviderException 
	 */
	public static X509Certificate getX509Certificate(InputStream inStream) throws CertificateException, NoSuchProviderException {
		if (inStream == null) {
			throw new IllegalArgumentException("inStream must not be null");
		}
		X509Certificate x509Cert = null;
		Certificate cert = CertificateFactory.getInstance("X.509", "BC").generateCertificate(inStream);
		if (cert instanceof X509Certificate) {
			x509Cert = (X509Certificate) cert;
		} else {
			CertUtils.LOGGER.warn("Certificate is not of type 'X509Certificate'.");
		}
		return x509Cert;
	}

	/**
	 * @param certBase64
	 *            String
	 * @return X509Certificate
	 * @throws CertificateException
	 *             CertificateException
	 * @throws NoSuchProviderException 
	 */
	public static X509Certificate getX509CertificateFromBase64(String certBase64) throws CertificateException, NoSuchProviderException {
		if (certBase64 == null || certBase64.equals("")) {
			throw new IllegalArgumentException("certBase64 must not be null or empty");
		}
		return CertUtils.getX509Certificate(Base64.decodeBase64(certBase64));
	}

	private static String getCN(X500Principal x500Principal) {
        
        String cn = null;
//      ZKSA-2285
//      bug in BC > 1.76: IllegalArgumentException when initializing X500Name with RFC1779 formatted distinguished name containing quotes  
        String nameRFC1779 = x500Principal.getName(X500Principal.RFC1779);
        String nameRFC2253 = x500Principal.getName(X500Principal.RFC2253);
        LOGGER.debug("RFC1779 formatted distinguished name: " + nameRFC1779);
        LOGGER.debug("RFC2253 formatted distinguished name: " + nameRFC2253);
        X500Name x500Name = null;
        try {
        	LOGGER.debug("trying RFC1779 formatted distinguished name");
        	x500Name = new X500Name(nameRFC1779);
        } catch (Exception e) {
        	LOGGER.debug("Error initializing X500Name with RFC1779 formatted name", e);
        }
        if (x500Name == null || x500Name.getRDNs(BCStyle.CN).length == 0) {
	        try {
	        	LOGGER.debug("trying RFC2253 formatted distinguished name");
	        	x500Name = new X500Name(nameRFC2253);
	        } catch (Exception e1) {
	        	LOGGER.error("Error initializing X500Name with RFC2253 formatted name", e1);
	        	return null;
	        }
        }
        RDN[] rdn = x500Name.getRDNs(BCStyle.CN);
        if (rdn.length == 0) {
        	LOGGER.error("No CN found in X.500 distinguished name");
        	return null;
        } 
       	cn = rdn[0].getFirst().getValue().toString();
       	LOGGER.debug("CN=" + cn);
        return cn;
	}
	
	/**
	 * Liefert den Subject-CN aus dem Zertifikat.
	 * 
	 * @param x509Cert
	 *            X509Certificate
	 * @return String
	 * @throws BMUException
	 *             BMUException
	 */
	public static String getSubjectCN(X509Certificate x509Cert) throws BMUException {
		if (x509Cert == null) {
			throw new IllegalArgumentException("x509Cert must not be null or empty");
		}
		X500Principal x500Principal = x509Cert.getSubjectX500Principal();
        String cn = getCN(x500Principal);
		
        CertUtils.LOGGER.debug("<getSubjectCN> cn=" + cn);
		return cn;
	}

	/**
	 * Liefert den Issuer-CN aus dem Zertifikat.
	 * 
	 * @param x509Cert
	 *            X509Certificate
	 * @return String
	 * @throws BMUException
	 *             BMUException
	 */
	public static String getIssuerCN(X509Certificate x509Cert) throws BMUException {
		if (x509Cert == null) {
			throw new IllegalArgumentException("x509Cert must not be null or empty");
		}
		X500Principal x500Principal = x509Cert.getIssuerX500Principal();
        String cn = getCN(x500Principal);
		
		CertUtils.LOGGER.debug("<getIssuerCN> cn=" + cn);
		return cn;
	}

//	/**
//	 * Liefert die OCSP-URL aus einem Zertifikat.
//	 * BC bis 1.47 !
//	 * @param certificate
//	 * @return Die OCSP-URL
//	 * @throws BMUException
//	 */
//	public static String getOcspUrl(X509Certificate certificate) throws BMUException {
//		String ocspURL = null;
//    	byte[] value =  certificate.getExtensionValue(X509Extensions.AuthorityInfoAccess.getId()); // "1.3.6.1.5.5.7.1.1"
//	    
//		AuthorityInformationAccess authorityInformationAccess;
//	    try {
//			ASN1InputStream is = new ASN1InputStream(new ByteArrayInputStream(value));
//	    	DEROctetString oct = (DEROctetString) is.readObject();
//	    	is.close();
//	    	is = new ASN1InputStream(oct.getOctets());
//			authorityInformationAccess = new AuthorityInformationAccess((org.bouncycastle.asn1.ASN1Sequence) is.readObject());
//			is.close();
//	    } catch (IOException e) {
//	    	LOGGER.error("Error extracting OCSP-URL from Certificate", e);
//	    	throw new BMUException("Error extracting OCSP-URL from Certificate", e);
//	    }
//		
//	    AccessDescription[] accessDescriptions = authorityInformationAccess.getAccessDescriptions();
//	    for (AccessDescription accessDescription : accessDescriptions) {
//	    	GeneralName gn = accessDescription.getAccessLocation();
//	    	DERIA5String str = DERIA5String.getInstance(gn.getDERObject());
//	    	ocspURL = str.getString();
//	    }
//	    
//    	LOGGER.debug("<getOcspUrl> URI: " + ocspURL);
//    	return ocspURL;
//    }
    
    
	/**
	 * Liefert die OCSP-URL aus einem Zertifikat.
	 * BC ab 1.48 !
	 * @param certificate
	 * @return Die OCSP-URL
	 * @throws BMUException
	 */
    public static String getOcspUrl(X509Certificate certificate) throws Exception {
        byte[] octetBytes = certificate
                .getExtensionValue(org.bouncycastle.asn1.x509.Extension.authorityInfoAccess.getId());

        DLSequence dlSequence = null;
        ASN1Encodable asn1Encodable = null;

        try {
            ASN1Primitive fromExtensionValue = JcaX509ExtensionUtils.parseExtensionValue(octetBytes);
            if (!(fromExtensionValue instanceof DLSequence))
                return null;
            dlSequence = (DLSequence) fromExtensionValue;
            for (int i = 0; i < dlSequence.size(); i++) {
                asn1Encodable = dlSequence.getObjectAt(i);
                if (asn1Encodable instanceof DLSequence)
                    break;
            }
            if (!(asn1Encodable instanceof DLSequence))
                return null;
            dlSequence = (DLSequence) asn1Encodable;
            for (int i = 0; i < dlSequence.size(); i++) {
                asn1Encodable = dlSequence.getObjectAt(i);
                if (asn1Encodable instanceof DERTaggedObject)
                    break;
            }
            if (!(asn1Encodable instanceof DERTaggedObject))
                return null;
            DERTaggedObject derTaggedObject = (DERTaggedObject) asn1Encodable;
            byte[] encoded = derTaggedObject.getEncoded();
            if (derTaggedObject.getTagNo() == 6) {
                int len = encoded[1];
                return new String(encoded, 2, len);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }
    
    public static void convertP12ToJks(File jksFile, File p12File, char[] pw) throws BMUException {
    	try {
			KeyStore p12 = KeyStore.getInstance("PKCS12");
			p12.load(new FileInputStream(p12File), pw);
			KeyStore jks = KeyStore.getInstance("JKS");
			jks.load(null, null);
			for (String alias : Collections.list(p12.aliases())) {
				if (p12.isKeyEntry(alias)) {
					Key key = p12.getKey(alias, pw);
					Certificate[] chain = p12.getCertificateChain(alias);
					jks.setKeyEntry(alias, key, pw, chain);
				}
			}
			FileOutputStream out = new FileOutputStream(jksFile);
			jks.store(out, pw);
			out.close();
		} catch (Exception e) {
			LOGGER.error("Fehler beim Umwandeln des KeyStore " + jksFile.getAbsolutePath(), e);
			throw new BMUException("Fehler beim Umwandeln des KeyStore " + jksFile.getAbsolutePath(), e);
		}
    }
}
