/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-10-14 17:45:48 +0000 (Mon, 14 Oct 2024) $ $Rev: 1844 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.xmlsec;

import org.apache.xml.security.Init;
import org.apache.xml.security.exceptions.XMLSecurityException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import de.consist.bmu.rule.schema.Namespace;
import de.consist.bmu.rule.util.CertUtils;
import de.consist.bmu.rule.util.XmlUtils;
import de.consist.bmu.rule.xpath.XPathFassade;

import org.apache.commons.codec.binary.Hex;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.interfaces.ECPublicKey;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.jce.spec.ECPublicKeySpec;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.Key;
import java.security.KeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import javax.xml.crypto.AlgorithmMethod;
import javax.xml.crypto.Data;
import javax.xml.crypto.KeySelector;
import javax.xml.crypto.KeySelectorException;
import javax.xml.crypto.KeySelectorResult;
import javax.xml.crypto.MarshalException;
import javax.xml.crypto.NodeSetData;
import javax.xml.crypto.XMLCryptoContext;
import javax.xml.crypto.XMLStructure;
import javax.xml.crypto.dsig.Reference;
import javax.xml.crypto.dsig.SignatureMethod;
import javax.xml.crypto.dsig.SignedInfo;
import javax.xml.crypto.dsig.XMLSignature;
import javax.xml.crypto.dsig.XMLSignatureException;
import javax.xml.crypto.dsig.XMLSignatureFactory;
import javax.xml.crypto.dsig.dom.DOMValidateContext;
import javax.xml.crypto.dsig.keyinfo.KeyInfo;
import javax.xml.crypto.dsig.keyinfo.KeyValue;
import javax.xml.crypto.dsig.keyinfo.RetrievalMethod;
import javax.xml.xpath.XPathExpressionException;

/**
 * Fassade um xmlsec.jar
 * 
 * Eigentlich wollten wir Java 6 einsetzen, weil dort gem JSR-105 das XML
 * Signature API Bestandteil der RT ist. Nun mssen wir die Library xmlsec.jar
 * selbst einbinden.
 * 
 * Die Methode pruefeSignaturen() liefert ein boolean zurck, dass Auskunft ber
 * den Status der Signaturprfung gibt.
 * 
 * Done: Alternative fr HexDumpEncoder einsetzen (z.B. Hex aus Apache Commons
 * Codec).
 * 
 * TODO Womglich validieren eines frisch geparsten Dokuments erforderlich
 * (Validierung und Defaultwerte); das hat sich wohl erledigt?
 * 
 * TODO Hinweis: Sicherheitsprobleme durch prparierte XML-Signaturen
 * http://www.heise.de/newsticker/meldung/93924/
 * 
 * TODO SHA1 ist eigentlich nicht mehr zulssig, aber es hngt vom Zertifikat
 * ab, ob andere Algorithmen untersttzt werden
 * 
 * FIXME MD5 msste eigentlich abgelehnt werden.
 * 
 * @author jannighoefer
 */
public final class XmlSecFassade {
    private static final Log LOGGER = LogFactory.getLog(XmlSecFassade.class);

    private static final String RSA_SHA256 = "http://www.w3.org/2001/04/xmldsig-more#rsa-sha256";
    private static final String RSA_SHA384 = "http://www.w3.org/2001/04/xmldsig-more#rsa-sha384";
    private static final String RSA_SHA512 = "http://www.w3.org/2001/04/xmldsig-more#rsa-sha512";
    private static final String RSA_RIPEMD160 = "http://www.w3.org/2001/04/xmldsig-more#rsa-ripemd160";

    private static final String XADES_NS = "http://uri.etsi.org/01903/v1.3.2#";

//    private static final String DEFAULT_XMLSIGFACTORY_14 = "org.jcp.xml.dsig.internal.dom.XMLDSigRI";
//    private static final String DEFAULT_XMLSIGFACTORY_15 = "org.apache.jcp.xml.dsig.internal.dom.XMLDSigRI";

    private static boolean _useJSR105 = false;

    private static XmlSecFassade _theInstance = null;
    private static boolean _bcFirst = false;

    private XMLSignatureFactory _signatureFactory = null;

    /**
     * XPath-Intersect-Filter.
     */
    public static final String XPATH_INTERSECT = "here()/ancestor::*[6]";

    /**
     * XPath-Subtract-Filter (schliesst vorhandene Signatur ein).
     */
    public static final String XPATH_SUBTRACT_INCLUDE_PREVIOUS_SIG = "here()/ancestor::*[5]/following-sibling::*[namespace-uri()='http://www.w3.org/2000/09/xmldsig#' and local-name()='Signature']";

    /**
     * XPath-Subtract-Filter (schliesst vorhandene Signatur nicht ein).
     */
    public static final String XPATH_SUBTRACT_EXCLUDE_PREVIOUS_SIG = "here()/ancestor::*[6]/*[namespace-uri()='http://www.w3.org/2000/09/xmldsig#' and local-name()='Signature']";

    /**
     * 
     */
    private XmlSecFassade() {
    }

    /**
     * @return XmlSecFassade
     */
    public static synchronized XmlSecFassade getInstance() {
        if (_theInstance == null) {
            _theInstance = new XmlSecFassade();
            if (!_theInstance.init()) {
                LOGGER.error("Fehler beim Initialisieren von XmlSec");
                _theInstance = null;
            }
        }
        return _theInstance;
    }

    /**
     * @return Gibt an, ob die JSR 105 (Java XML Digital Signature) API bei der
     *         Validierung der Signaturen verwendet wird
     */
    public static boolean isUseJSR105() {
        return _useJSR105;
    }

    public static boolean isBcFirst() {
        return _bcFirst;
    }
    /**
     * @param useJSR105
     *            Gibt an, ob die JSR 105 (Java XML Digital Signature) API bei
     *            der Validierung der Signaturen verwendet werden soll
     */
    public static void setUseJSR105(boolean useJSR105) {
        _useJSR105 = useJSR105;
    }

    /**
     * Allgemeine Initialisierung von XMLSEC.
     * 
     * @return boolean
     */
    public synchronized boolean init() {
        if (_signatureFactory != null) {
            return true;
        }

        boolean binitialized = true;
        String javaVersion = System.getProperty("java.version");
        boolean bcFirst = javaVersion.startsWith("1.");

        if (loadBouncyCastle()) {
            LOGGER.debug("BouncyCastle ist initialisiert");
        } else {
            LOGGER.error("BouncyCastle konnte nicht geladen werden.");
        }
        Provider[] providers = Security.getProviders();
        if (providers != null && providers.length > 0) {
            if ("BC".equals(providers[0].getName())) {
                if (bcFirst) {
	            	// Alles Ok
	                LOGGER.debug("BouncyCastle " + providers[0].getName()
	                        + ", Version "
	                        + providers[0].getVersionStr());
//                    		+ providers[0].getVersionStr());
                } else {
                	LOGGER.debug("BouncyCastle ist der erste Provider in der Liste mit Java > 1.8");
                }
            } else {
                if (bcFirst) {
                	LOGGER.debug("BouncyCastle ist nicht der erste Provider in der Liste.");
                } else {
	            	// Alles Ok
	                LOGGER.debug("Provider " + providers[0].getName()
	                        + ", Version "
	                        + providers[0].getVersionStr());
//                    		+ providers[0].getVersionStr());
                }
            }
        } else {
            LOGGER.error("Liste der Provider kann nicht abgefragt werden.");
        }
        // XMLSec initialisieren?
        if (!Init.isInitialized()) {
            LOGGER.debug("Initialisiere XMLSEC");
            System.setProperty("org.apache.xml.security.resource.config", "security-config-xalan.xml");
            Init.init();
            LOGGER.debug("XMLSEC ist initialisiert");
        }

        if (LOGGER.isDebugEnabled()) {
//            boolean testInternalsVersions = true;
            LOGGER.debug("Java version: " + javaVersion);
            // 1.7.0_80, 1.8.0_161, 9.0.4, 10.0.1
//            if (!javaVersion.startsWith("1.")) {
//                LOGGER.debug("Modulare Java Version, teste die internen Xerces und Xalan nicht.");
//                testInternalsVersions = false;
//            }
            // _logger.debug("Xerces version: " +
            // org.apache.xerces.impl.Version.getVersion());
//            LOGGER.debug("Xerces version: "
//                    + org.apache.xerces.impl.Version.getVersion());
//            if (testInternalsVersions) {
//                String value = callGetVersion("com.sun.org.apache.xerces.internal.impl.Version");
//                LOGGER.debug("Xerces internal version: " + value);
//            }
//            LOGGER.debug("Xalan version: "
//                    + org.apache.xalan.Version.getVersion());
//            if (testInternalsVersions) {
//                String value = callGetVersion("com.sun.org.apache.xalan.internal.Version");
//                LOGGER.debug("Xalan internal version: " + value);
//            }
        }

        // Provider ber Systemparameter auslesen:
//        String providerName = System.getProperty("jsr105Provider");
//        if (providerName == null) {
//            providerName = DEFAULT_XMLSIGFACTORY_15;
//        } else {
//            LOGGER.debug("Using XmlDSig provider from system property 'jsr105Provider: "
//                    + providerName);
//        }
//        try {
//            Class.forName(providerName);
//        } catch (ClassNotFoundException e1) {
//            providerName = DEFAULT_XMLSIGFACTORY_14;
//            LOGGER.debug("Provider for XmlSec-1.5.x not found, trying 1.4.x :"
//                    + providerName);
//            try {
//                Class.forName(providerName);
//            } catch (ClassNotFoundException e) {
//                LOGGER.error("XMLDSig provider class not found: "
//                        + providerName);
//                binitialized = false;
//            }
//        }

        // Neue Signierungsfactory erzeugen:
//        try {
//            LOGGER.debug("Instanziiere " + providerName);
//            Provider provider = (Provider) Class.forName(providerName).getDeclaredConstructor()
//                    .newInstance();
//            LOGGER.debug("JSR105-Provider " + provider.getName() + ", Version "
//                    + Double.toString(provider.getVersion()));
////            		+ provider.getVersionStr());
//            _signatureFactory = XMLSignatureFactory
//                    .getInstance("DOM", provider);
//            LOGGER.debug("XMLSignatureFactory fr Typ '"
//                    + _signatureFactory.getMechanismType() + "' initialisiert.");
//        } catch (Exception e) {
//            binitialized = false;
//            LOGGER.error("<init>", e);
//        }
        _signatureFactory = XMLSignatureFactory.getInstance("DOM");

        Package pkg = ClassLoader.getPlatformClassLoader().getDefinedPackage("xmlsec");
//        Package pkg = ClassLoader.getSystemClassLoader().getDefinedPackage("xmlsec");

        if (pkg != null) {
            LOGGER.debug(pkg.getImplementationTitle() + ", version: "
                    + pkg.getImplementationVersion());
        } else {
            LOGGER.debug("<init> Keine Versionsinformation fr XmlSec gefunden!");
        }

        return binitialized;
    }

    private static boolean checkBrainpoolP256r1() {
        String[] curves = Security.getProvider("SunEC").getProperty("AlgorithmParameters.EC SupportedCurves").split("\\|");
        for (String curve : curves) {
            if (curve.contains("1.3.36.3.3.2.8.1.1.7")) {
                LOGGER.debug("brainpoolP256r1 supported by SunEC");
                return true;
            }
        }
        LOGGER.info("Die elliptische Kurve 'brainpoolP256r1' wird vom SunEC Provider nicht untersttzt.");
        return false;
    }
    
    /**
     * Ldt den Security-Provider BouncyCastle.
     * 
     * @return boolean
     */
    public static boolean loadBouncyCastle() {
        if (Security.getProvider("BC") == null) {
            // Instantiate the BouncyCastle provider
            try {
                // In Java 21 und hher ist der truststore 'cacerts' vom Typ PKCS12, beim Versuch diesen mit dem default-pw 'changeit' zu laden wird eine IOException
                // (password supplied for keystore that does not require one) geworfen. Das Setzen der folgenden Property verhindert das. 
                System.setProperty("org.bouncycastle.pkcs12.ignore_useless_passwd", "true");
                
                Class<?> bcProvClass = Class
                        .forName("org.bouncycastle.jce.provider.BouncyCastleProvider");
                Provider bcProv = (Provider) bcProvClass.getDeclaredConstructor().newInstance();

                String javaVersion = System.getProperty("java.version");
                String javaVendor = System.getProperty("java.vendor");
                LOGGER.info("Java version: " + javaVersion + ", vendor: " + javaVendor);
                // 1.7.0_80, 1.8.0_161, 9.0.4, 10.0.1
                boolean brainPoolSupported = checkBrainpoolP256r1();
                // Java Version <= 1.8
                if (javaVersion.startsWith("1.") || !brainPoolSupported /*javaVersion.contains("redhat")*/) {
	                if (Security.insertProviderAt(bcProv, 1) > -1) {
	                    LOGGER.info("BouncyCastle Provider wurde an Position 1 eingefgt.");
	                    _bcFirst = true;
	                    return true;
	                }
                } else {
		            if (Security.addProvider(bcProv) > -1) {
	                    LOGGER.info("BouncyCastle Provider wurde hinzugefgt.");
		                _bcFirst = false;
		            	return true;
		            }
                }
            } catch (Exception e) {
                LOGGER.error("BouncyCastle nicht gefunden!", e);
            }
        } else {
            LOGGER.info("BouncyCastle Provider ist bereits vorhanden.");
            return true;
        }
        // Exception ist aufgetreten
        return false;
    }

    /**
     * 
     * 
     * @param doc
     *            das Document, dass die zu prfenden Signaturen enthlt
     * @param signatureMustExist
     *            legt fest, ob Signaturen vorhanden sein mssen
     * @param sigID
     *            String
     * @return true, wenn Signaturen gltig, false wenn Signaturen ungltig
     * @throws XmlSecException
     *             XmlSecException
     * @deprecated Deutsche Funktionsnamen sind nicht professionell
     */
    @Deprecated
    public boolean pruefeSignaturen(Document doc, boolean signatureMustExist,
            String sigID) throws XmlSecException {
        if (doc == null) {
            throw new IllegalArgumentException("Document must not be null");
        }

        try {
            return pruefeSignaturen0(doc, signatureMustExist, sigID);
        } catch (MarshalException e) {
            LOGGER.error("<pruefeSignaturen>", e);
            throw new XmlSecException(e);
        } catch (XMLSignatureException e) {
            LOGGER.error("<pruefeSignaturen>", e);
            throw new XmlSecException(e);
        } catch (Exception e) {
            LOGGER.error("<pruefeSignaturen>", e);
            throw new XmlSecException(e);
        }
    }

    /**
     * @param doc
     *            Document
     * @return List
     * @throws XmlSecException
     *             XmlSecException
     */
    public List<String> verifySignatures(Document doc) throws XmlSecException {
        if (doc == null) {
            throw new IllegalArgumentException("Document must not be null");
        }
        List<String> sigIdInvalidList = new ArrayList<String>();
        NodeList nl = doc.getElementsByTagNameNS(XMLSignature.XMLNS,
                "Signature");
        LOGGER.debug("Prfe " + nl.getLength() + " Signatur(en)");
        for (int i = 0; i < nl.getLength(); i++) {
            Node sigNode = nl.item(i);
            Node idNode = sigNode.getAttributes().getNamedItem("Id");
            String sigId = null;
            if (idNode != null) {
                sigId = idNode.getTextContent();
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Prfe Signatur " + i + "/" + nl.getLength()
                            + " mit Id='" + sigId + "'");
                }
                if (!verifySignature(doc, sigId)) {
                    sigIdInvalidList.add(sigId);
                }
            }
        }
        return sigIdInvalidList;
    }

    /**
     * @param doc
     *            Document
     * @param sigId
     *            String
     * @return boolean
     * @throws XmlSecException
     *             XmlSecException
     */
    public boolean verifySignature(Document doc, String sigId)
            throws XmlSecException {
        boolean valid = true;
        if (doc == null) {
            throw new IllegalArgumentException("No document given.");
        }

        Node sigNode = null;
        String sigXPath = "descendant::ds:Signature[@Id='" + sigId + "']";

        try {
            sigNode = XPathFassade.getInstance().evaluateNode(doc, sigXPath);
        } catch (XPathExpressionException e) {
            LOGGER.error("error evaluating XPath: " + sigXPath);
            throw new XmlSecException("error evaluating XPath: " + sigXPath, e);
        }

        if (_useJSR105) {

            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("using JSR 105 API..");
            }

            DOMValidateContext valContext = new DOMValidateContext(
                    new KeyValueKeySelector(), sigNode);
            valContext.setProperty("javax.xml.crypto.dsig.cacheReference",
                    Boolean.TRUE);

            // unmarshal the XMLSignature
            XMLSignature signature = null;
            try {
                synchronized (_signatureFactory) {
                    signature = _signatureFactory
                            .unmarshalXMLSignature(valContext);
                }
            } catch (MarshalException e) {
                LOGGER.debug("unmarshal signature failed: " + e.getMessage());
                LOGGER.debug("trying apache xmlsec implementation..");
                if (!verifySignatureApache((Element) sigNode)) {
                    valid = false;
                }
            }

            if (signature != null) {
                try {
                    // ID Attribute fuer Xades Reference setzen..
                    prepareXadesReferenceId((Element) sigNode);

                    // Validate the XMLSignature (generated above)
                    boolean coreValidity = signature.validate(valContext);

                    // Check core validation status
                    if (!coreValidity) {
                        LOGGER.error("Signatur mit ID '" + signature.getId()
                                + "' ist NICHT gltig!");
                        valid = false;

                        boolean sv = signature.getSignatureValue().validate(
                                valContext);
                        LOGGER.debug("signature value validation status: " + sv);

                        // check the validation status of each Reference

                        SignedInfo si = signature.getSignedInfo();
                        Iterator<Reference> iter = si.getReferences()
                                .iterator();

                        for (int j = 0; iter.hasNext(); j++) {
                            Reference ref = iter.next();
                            boolean refValid = ref.validate(valContext);
                            LOGGER.debug("ref[" + j + "], id '" + ref.getId()
                                    + "' validity status: " + refValid);
                            if (!refValid) {
                                dumpReference(ref/* , valContext */);
                            }
                        }
                    } else {
                        LOGGER.debug("Signatur mit ID '" + signature.getId()
                                + "' ist gltig");
                    }
                } catch (XMLSignatureException e) {
                    LOGGER.error("<pruefeSignaturen>", e);
                    throw new XmlSecException(e);
                }
            }
        } else {

            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("using Apache Santuario API..");
            }

            valid = verifySignatureApache((Element) sigNode);
        }
        return valid;
    }

    /**
     * Check Signatures using apache xmlsec implementation.
     * 
     * @param signElement
     *            Element
     * @return true, wenn Signaturen gltig, false wenn Signaturen ungltig
     * @throws XmlSecException
     *             XmlSecException
     */
    public boolean verifySignatureApache(Element signElement)
            throws XmlSecException {
        boolean result = false;
        long start = System.currentTimeMillis();
        org.apache.xml.security.signature.XMLSignature signature = null;
        PublicKey pk = null;
        try {
            // ID Attribute fuer Xades Reference setzen..
            prepareXadesReferenceId(signElement);

            signature = new org.apache.xml.security.signature.XMLSignature(
                    signElement, "");

            org.apache.xml.security.keys.KeyInfo ki = signature.getKeyInfo();
            // result =
            // sign.checkSignatureValue(ki.getX509Certificate().getPublicKey());
            pk = ki.getPublicKey();
            result = signature.checkSignatureValue(pk);
            if (result) {
                LOGGER.debug("Signature passed core validation!");
//                if (LOGGER.isTraceEnabled()) {
//                    org.apache.xml.security.signature.SignedInfo si = signature
//                            .getSignedInfo();
//                    boolean refStatus = si.verifyReferences();
//                    LOGGER.trace("Reference validation status: " + refStatus);
//                    for (int i = 0; i < si.getLength(); i++) {
//                        org.apache.xml.security.signature.Reference ref = (org.apache.xml.security.signature.Reference) si
//                                .item(i);
//                        boolean refValid = ref.verify();
//                        LOGGER.trace("ref[" + i + "] validity status: "
//                                + refValid);
//                        dumpReference(ref);
//                    }
//                }
            } else {
                org.apache.xml.security.signature.SignedInfo si = signature
                        .getSignedInfo();
                LOGGER.warn("Signature failed core validation!");
                boolean refStatus = si.verifyReferences();
                LOGGER.debug("Reference validation status: " + refStatus);
                if (!refStatus) {
                    for (int i = 0; i < si.getLength(); i++) {
                        org.apache.xml.security.signature.Reference ref = (org.apache.xml.security.signature.Reference) si
                                .item(i);
                        boolean refValid = ref.verify();
                        LOGGER.debug("ref[" + i + "], id '" + ref.getId()
                                    + "' validity status: " + refValid);
                    }
                }
            }
        } catch (XMLSecurityException e) {
            LOGGER.error("error verifying signature", e);
            throw new XmlSecException("error verifying signature", e);
        }
        if (LOGGER.isTraceEnabled()) {
            long dauer = System.currentTimeMillis() - start;
            LOGGER.trace("Dauer der Prfung: "
                    + (dauer > 5000 ? dauer / 1000 + "s." : dauer + "ms."));
        }

        return result;
    }

   public PublicKey unmarshalECDSAKeyValue(Element kvtElem)
            throws MarshalException {
        Element curElem = XmlUtils.getFirstChildElement(kvtElem);
        if (curElem == null) {
            throw new MarshalException(
                    "KeyValue must contain at least one type");
        }

        String oid = null;
        String ecdsaNS = Namespace.xmldsig_more.getUri();
        if (curElem.getLocalName().equals("ECParameters")
                && ecdsaNS.equals(curElem.getNamespaceURI())) {
            throw new UnsupportedOperationException(
                    "ECParameters not supported");
        } else if (curElem.getLocalName().equals("NamedCurve")
                && ecdsaNS.equals(curElem.getNamespaceURI())) {
            String uri = XmlUtils.getAttributeValue(curElem, "URN");
            if (uri.startsWith("urn:oid:")) {
                oid = uri.substring(8);
            } else {
                throw new MarshalException("Invalid NamedCurve URI");
            }
        } else {
            throw new MarshalException("Invalid ECKeyValue");
        }
        curElem = XmlUtils.getNextSiblingElement(kvtElem);
        PublicKey pk = null;
        PublicKey pk1 = null;
        try {
            Element publicKeyXElem = XmlUtils.getChildElement(curElem, ecdsaNS, "X");
            String publicKeyXStr = publicKeyXElem.getAttributeNS(null, "Value");
            Element publicKeyYElem = XmlUtils.getChildElement(curElem, ecdsaNS, "Y");
            String publicKeyYStr = publicKeyYElem.getAttributeNS(null, "Value");
            pk = getECPublicKey(publicKeyXStr, publicKeyYStr, oid);
            pk1 = generatePublicKey(new BigInteger(publicKeyXStr, 10), new BigInteger(publicKeyYStr, 10), oid);
            if (!pk.equals(pk1)) {
                LOGGER.error("keys differ!");
            }
        } catch (Exception e) {
            LOGGER.error("error getting public key", e);
        }
        return pk;
    }

    private PublicKey generatePublicKey(BigInteger x, BigInteger y,
            String sCurve) throws IOException, NoSuchAlgorithmException,
            NoSuchProviderException, InvalidKeySpecException {
        ECNamedCurveParameterSpec spec = ECNamedCurveTable
                .getParameterSpec(sCurve);
        ECPoint pPublicPoint = spec.getCurve().createPoint(x, y);
        ECPublicKeySpec publicSpec = new ECPublicKeySpec(pPublicPoint, spec);
        KeyFactory keyfac = KeyFactory.getInstance("ECDSA", "BC");
        return keyfac.generatePublic(publicSpec);
    }

    private PublicKey getECPublicKey(String x, String y, String curve)
            throws NoSuchAlgorithmException, InvalidKeySpecException,
            NoSuchProviderException {
        ECNamedCurveParameterSpec spec = ECNamedCurveTable
                .getParameterSpec(curve);
//        ECNamedCurveSpec params = new ECNamedCurveSpec(curve, spec.getCurve(),
//                spec.getG(), spec.getN());
//        EllipticCurve eCurve = params.getCurve();
        //ECCurve.Fp ecCurve = (Fp) spec.getCurve();
        ECCurve ecCurve = spec.getCurve();
        //BigInteger q = ecCurve.getQ();
        //ECPoint ecPoint = ecCurve.createPoint(new BigInteger(x, 10), new BigInteger(y, 10), false);
//        ECFieldElement x1 = ecCurve.fromBigInteger(new BigInteger(x));
//        ECFieldElement y1 = ecCurve.fromBigInteger(new BigInteger(y));
        ECPoint ecPoint = ecCurve.createPoint(new BigInteger(x), new BigInteger(y));
        //org.bouncycastle.math.ec.ECPoint ecPoint = org.bouncycastle.math.ec.ECPoint(ecCurve, x1, y1);
        // ECField ecf = eCurve.getField();
        // ECFieldFp ecpf = new ECFieldFp(new BigInteger(x));
        // ECPoint point = new ECPoint(new BigInteger(x), new BigInteger(y));
        ECPublicKeySpec pubKeySpec = new ECPublicKeySpec(ecPoint, spec);
        KeyFactory kf = KeyFactory.getInstance("EC", "BC");
        ECPublicKey pk = (ECPublicKey) kf.generatePublic(pubKeySpec);
        return pk;
    }
   
    /**
     * @param sigElem
     *            Signatur-Element
     * @return true, wenn das X509Certificate zum KeyValue passt, sonst false
     */
    public boolean validateX509Certificate(Element sigElem) {
        boolean valid = false;
        org.apache.xml.security.signature.XMLSignature signature = null;
        try {
            signature = new org.apache.xml.security.signature.XMLSignature(
                    sigElem, "");

            org.apache.xml.security.keys.KeyInfo ki = signature.getKeyInfo();
            if (ki != null) {
                LOGGER.trace("containsKeyValue: " + ki.containsKeyValue());
                org.apache.xml.security.keys.content.KeyValue kv = ki
                        .itemKeyValue(0);
                if (kv != null) {
                    Element kvElem = XmlUtils.getFirstChildElement(kv.getElement());
                    PublicKey pk = null;
                    if ("ECDSAKeyValue".equals(kvElem.getLocalName())
                            && Namespace.xmldsig_more.getUri().equals(
                                    kvElem.getNamespaceURI())) {
                        pk = unmarshalECDSAKeyValue(XmlUtils.getFirstChildElement(kvElem)); 
//                        LOGGER.warn("PublicKey from ECDSAKeyValue is currently not supported!");
//                        pk = kv.getPublicKey();
//                        valid = true;
                    } else {
                        pk = ki.getPublicKey();
                    }
                    if (pk != null && ki.getX509Certificate() != null && ki.getX509Certificate().getPublicKey() != null) {
                        PublicKey pkCert = ki.getX509Certificate().getPublicKey();
//                        if (pkCert instanceof JCEECPublicKey && pk instanceof JCEECPublicKey) {
//                            JCEECPublicKey ecpk = (JCEECPublicKey) pk;
//                            JCEECPublicKey ecpkCert = (JCEECPublicKey) pkCert;
//                            LOGGER.trace("Q: " + ecpk.getQ().equals(ecpkCert.getQ()));
//                            LOGGER.trace("parameters: " + ecpk.getParameters().equals(ecpkCert.getParameters()));
//                            LOGGER.trace("W: " + ecpk.getW().equals(ecpkCert.getW()));
//                            LOGGER.trace("Q(engine): " + ecpk.engineGetQ().equals(ecpkCert.engineGetQ()));
//                            LOGGER.trace("Algorithm: " + ecpk.getAlgorithm().equals(ecpkCert.getAlgorithm()));
//                        }
                        byte[] key = pk.getEncoded();
                        byte[] certKey = pkCert.getEncoded();
                        LOGGER.trace("key from 'ds:KeyValue:        "
                                + Hex.encodeHexString(key));
                        LOGGER.trace("key from 'ds:X509Certificate: "
                                + Hex.encodeHexString(certKey));
                        valid = Arrays.equals(key, certKey);
                    } else if (!valid) {
                        LOGGER.error("PublicKey and/or X509Certificate is not accessible");
                    }
                } else {
                    LOGGER.error("KeyValue not accessible");
                }
            } else {
                LOGGER.error("KeyInfo not accessible");
            }
        } catch (XMLSecurityException e) {
            LOGGER.error("error validating public key", e);
        } catch (MarshalException e) {
            LOGGER.error("error validating public key", e);
        } catch (Throwable e) {
            LOGGER.error("error validating public key", e);
        }
        return valid;
    }

    /**
     * @param doc
     * @param signatureMustExist
     * @param sigID
     * @return true, wenn Signaturen gltig, false wenn Signaturen ungltig
     * @throws MarshalException
     * @throws XMLSignatureException
     * @deprecated Deutsche Funktionsnamen sind nicht professionell
     */
    @Deprecated
    protected boolean pruefeSignaturen0(Document doc,
            boolean signatureMustExist, String sigID) throws MarshalException,
            XMLSignatureException, XmlSecException {
        boolean valid = true;

        if (doc == null) {
            throw new IllegalArgumentException("No document given.");
        }

        NodeList nl = doc.getElementsByTagNameNS(XMLSignature.XMLNS,
                "Signature");

        if ((nl.getLength() == 0) && signatureMustExist) {
            valid = false;
            LOGGER.warn("Cannot find Signature element");

            // throw new Exception("Cannot find Signature element");
        }

        // Create a DOM XMLSignatureFactory that will be used to unmarshal the
        // document containing the XMLSignature
        // XMLSignatureFactory fac = XMLSignatureFactory.getInstance("DOM");
        for (int i = 0; i < nl.getLength(); i++) {
            // Create a DOMValidateContext and specify a KeyValue
            // KeySelector
            // and document context
            Node sigNode = nl.item(i);
            // td.dump(sigNode);

            DOMValidateContext valContext = new DOMValidateContext(
                    new KeyValueKeySelector(), nl.item(i));
            valContext.setProperty("javax.xml.crypto.dsig.cacheReference",
                    Boolean.TRUE);

            // unmarshal the XMLSignature
            XMLSignature signature = null;
            try {
                signature = _signatureFactory.unmarshalXMLSignature(valContext);
            } catch (MarshalException e) {
                LOGGER.debug("unmarshal signature failed: " + e.getMessage());
                LOGGER.debug("trying apache xmlsec implementation..");
                if (!verifySignatureApache((Element) sigNode)) {
                    valid = false;
                }
                continue;
            }

            if (sigID == null || sigID.equals(signature.getId())) {

                LOGGER.debug("Prfe Signatur mit ID=" + signature.getId());

                // Validate the XMLSignature (generated above)
                boolean coreValidity = signature.validate(valContext);

                SignedInfo si = signature.getSignedInfo();
                Iterator<Reference> iter = null;

                if (LOGGER.isDebugEnabled()) {
                    iter = si.getReferences().iterator();

                    for (int j = 0; iter.hasNext(); j++) {
                        Reference ref = iter.next();
                        boolean refValid = ref.validate(valContext);
                        LOGGER.debug("ref[" + j + "], id '" + ref.getId()
                                + "' validity status: " + refValid);
                        // if (!refValid) {
                        dumpReference(ref/* , valContext */);
                        // }
                    }
                }

                // Check core validation status
                if (!coreValidity) {
                    LOGGER.error("Signatur mit ID '" + signature.getId()
                            + "' ist NICHT gltig!");
                    valid = false;

                    boolean sv = signature.getSignatureValue().validate(
                            valContext);
                    LOGGER.debug("signature value validation status: " + sv);
                    // if (!sv && _logger.isDebugEnabled()) {
                    // _logger.debug("signature value:\n" + new
                    // HexDumpEncoder().encode(signature.getSignatureValue().getValue()));
                    // }
                    // check the validation status of each Reference
                    iter = si.getReferences().iterator();

                    for (int j = 0; iter.hasNext(); j++) {
                        Reference ref = iter.next();
                        boolean refValid = ref.validate(valContext);
                        if (LOGGER.isDebugEnabled()) {
                            LOGGER.debug("ref[" + j + "], id '" + ref.getId()
                                    + "' validity status: " + refValid);
                            if (!refValid) {
                                dumpReference(ref/* , valContext */);
                            }
                        }
                    }
                } else {
                    LOGGER.debug("Signatur mit ID '" + signature.getId()
                            + "' ist gltig");
                }
            }
        }

        return valid;
    }

    private void dumpReference(Reference ref/* , XMLCryptoContext context */) {
        // HexDumpEncoder hde = new HexDumpEncoder();
        // Hex hde = new Hex();
        LOGGER.debug("dumping reference: " + ref.getId() + ", URI: "
                + ref.getURI());

        byte[] dv = ref.getDigestValue();

        if (dv != null) {
            LOGGER.debug("digestvalue: " + Hex.encodeHexString(dv));
        } else {
            LOGGER.debug("no digest value");
        }

        InputStream is = ref.getDigestInputStream();

        if (is != null) {
            try {
                byte[] data = new byte[is.available()];
                is.read(data);
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("digest input data:\n" + new String(data));
                }
            } catch (IOException e) {
                LOGGER.error("<dumpReference>", e);
            }
        } else {
            LOGGER.debug("digest input stream of refernce is null");
        }
    }

//    private String callGetVersion(String name) {
//        String retVal = null;
//        @SuppressWarnings("rawtypes")
//        final Class[] noArgs = new Class[0];
//        try {
//            @SuppressWarnings("rawtypes")
//            Class clazz = Class.forName(name, false,
//                    XmlSecFassade.class.getClassLoader());
//            @SuppressWarnings("unchecked")
//            Method method = clazz.getMethod("getVersion", noArgs);
//            retVal = method.invoke(null, new Object[0]).toString();
//        } catch (ClassNotFoundException e) {
//            e.printStackTrace(System.err);
//            retVal = "class not found";
//        } catch (NoSuchMethodException e) {
//            e.printStackTrace(System.err);
//            retVal = "method not found";
//        } catch (InvocationTargetException e) {
//            e.printStackTrace(System.err);
//            retVal = "invocation target exception";
//        } catch (IllegalAccessException e) {
//            e.printStackTrace(System.err);
//            retVal = "illegal access exception";
//        }
//        return retVal;
//    }

    /**
     * KeySelector which retrieves the public key out of the KeyValue element
     * and returns it. NOTE: If the key algorithm doesn't match signature
     * algorithm, then the public key will be ignored.
     */
    private static class KeyValueKeySelector extends KeySelector {
        public KeySelectorResult select(KeyInfo keyInfo,
                KeySelector.Purpose purpose, AlgorithmMethod method,
                XMLCryptoContext context) throws KeySelectorException {
            if (keyInfo == null) {
                throw new KeySelectorException("Null KeyInfo object!");
            }

            if (method == null) {
                throw new KeySelectorException("Null KeyInfo object!");
            }
            SignatureMethod sm = (SignatureMethod) method;
            List<XMLStructure> list = keyInfo.getContent();

            for (int i = 0; i < list.size(); i++) {
                XMLStructure xmlStructure = (XMLStructure) list.get(i);

                if (xmlStructure instanceof KeyValue) {
                    LOGGER.debug("XMLStructure of type 'KeyValue' found.");
                    PublicKey pk = null;

                    try {
                        pk = ((KeyValue) xmlStructure).getPublicKey();
                    } catch (KeyException ke) {
                        LOGGER.error("KeyException: " + ke.getMessage());
                        throw new KeySelectorException(ke);
                    }

                    // make sure algorithm is compatible with method
                    if (algEquals(sm.getAlgorithm(), pk.getAlgorithm())) {
                        return new SimpleKeySelectorResult(pk);
                    } else {
                        LOGGER.error("Algorithm of KeyValue not compatible: "
                                + pk.getAlgorithm() + ", expected: "
                                + sm.getAlgorithm());
                    }
                } else if (xmlStructure instanceof RetrievalMethod) {
                    RetrievalMethod rm = (RetrievalMethod) xmlStructure;
                    LOGGER.debug("RetrievalMethod found, URI: " + rm.getURI());
                    try {
                        Data derefData = rm.dereference(context);
                        if (derefData instanceof NodeSetData) {
                            NodeSetData<?> nds = (NodeSetData<?>) derefData;
                            Iterator<?> iter = nds.iterator();
                            while (iter.hasNext()) {
                                Object o = iter.next();
                                if (o instanceof Element) {
                                    String certb64 = XPathFassade.getInstance()
                                            .evaluate((Element) o,
                                                    "//ds:X509Certificate");
                                    /*
                                     * byte[] certData = Base64.decode(certb64);
                                     * CertificateFactory cf =
                                     * CertificateFactory .getInstance("X509");
                                     * ByteArrayInputStream bais = new
                                     * ByteArrayInputStream( certData);
                                     * X509Certificate cert = (X509Certificate)
                                     * cf .generateCertificate(bais);
                                     */
                                    X509Certificate cert = CertUtils
                                            .getX509CertificateFromBase64(certb64);
                                    PublicKey pk = cert.getPublicKey();
                                    // make sure algorithm is compatible with
                                    // method
                                    if (algEquals(sm.getAlgorithm(),
                                            pk.getAlgorithm())) {
                                        return new SimpleKeySelectorResult(pk);
                                    } else {
                                        LOGGER.error("Algorithm of KeyValue not compatible: "
                                                + pk.getAlgorithm()
                                                + ", expected: "
                                                + sm.getAlgorithm());
                                    }
                                } else {
                                    LOGGER.error("Unexpected NodeSetData entry: "
                                            + ((o != null) ? o.getClass()
                                                    .getName() : "null"));
                                }
                            }
                        } else {
                            // skip; keyinfo type is not supported
                            LOGGER.error("KeyInfo type not supported: "
                                    + ((derefData != null) ? derefData
                                            .getClass().getName() : "null"));
                            continue;
                        }
                    } catch (Exception e) {
                        LOGGER.error("Error selecting KeyValue: "
                                + e.getMessage());
                        throw new KeySelectorException(e);
                    }
                } else {
                    LOGGER.debug("KeyValue select-method not supported: "
                            + ((xmlStructure != null) ? xmlStructure.getClass()
                                    .getName() : "null"));
                }
            }

            LOGGER.error("No KeyValue found!");
            throw new KeySelectorException("No KeyValue element found!");
        }

        // @@@FIXME: this should also work for key types other than DSA/RSA
        static boolean algEquals(String algURI, String algName) {
            if (algName.equalsIgnoreCase("DSA")
                    && algURI.equalsIgnoreCase(SignatureMethod.DSA_SHA1)) {
                return true;
            } else if (algName.equalsIgnoreCase("RSA")
                    && ((algURI.equalsIgnoreCase(SignatureMethod.RSA_SHA1)
                            || algURI.equalsIgnoreCase(RSA_SHA256)
                            || algURI.equalsIgnoreCase(RSA_SHA384)
                            || algURI.equalsIgnoreCase(RSA_SHA512) || algURI
                                .equalsIgnoreCase(RSA_RIPEMD160)))) {
                return true;
            }
            return false;
        }
    }

    private static class SimpleKeySelectorResult implements KeySelectorResult {
        private PublicKey _pk;

        SimpleKeySelectorResult(PublicKey pk) {
            this._pk = pk;
        }

        public Key getKey() {
            return _pk;
        }
    }

    private void prepareXadesReferenceId(Element signElement) {
        NodeList nlXadesSignedProperties = signElement.getElementsByTagNameNS(
                XADES_NS, "SignedProperties");
        if (nlXadesSignedProperties.getLength() > 0) {
            Element xadesSignedProperties = (Element) nlXadesSignedProperties
                    .item(0);
            if (xadesSignedProperties.hasAttribute("Id")) {
                xadesSignedProperties.setIdAttribute("Id", true);
            } else {
                LOGGER.debug("Attribute 'Id' not found on Element 'XadesSignedProperties'");
            }
        }
    }
}
